//Routines (read *.r format, save to XMLSS etc) for FlyLogReader
//Last update: 2012.05.30

unit osdroutines;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, StdCtrls, Graphics, zexmlss, zexmlssutils, zsspxml, zeodfs, zeformula;

type
  TFlyOSDRecord = packed record
    State: byte;                    //1
    Latitude: longint;              //4
    longitude: longint;             //4
    MSL_Altitude: longint;          //4
    Speed: word;                    //2
    Azimuth: word;                  //2
    DateTime: array [0..2] of byte; //3
    Temperature: byte;              //1
    RotateSpeed: word;              //2
    SYSBatVol: word;                //2
    SYSBatVideo: word;              //2
    SYSBatCur: word;                //2
    FlyMode: byte;                  //1
  end;                //total bytes: 30

  TFlyOSDArray = array of TFlyOSDRecord;

  TFlyOSDHeader = packed record
    iSWVer: word;                   //2
    iHWVer: word;                   //2
    time_year: smallint;            //2
    time_month: smallint;           //2
    time_day: smallint;             //2
    time_hour: smallint;            //2
    time_minute: smallint;          //2
    time_second: smallint;          //2
    timezone: shortint;             //1
  end;                //total bytes: 17

function ReadFlyOSDR(Stream: TStream; var FlyArray: TFlyOSDArray; var Count: integer; var FlyHeader: TFlyOSDHeader; var FormatVersion: integer; Memo: TMemo): byte; overload;
function ReadFlyOSDR(const FileName: string; var FlyArray: TFlyOSDArray; var Count: integer; var FlyHeader: TFlyOSDHeader; var FormatVersion: integer; Memo: TMemo): byte; overload;
function OnlyDots(const st: string): string;
function IntToStrN(value: integer; NullCount: integer): string;
procedure ArrayToXMLSS(var FlyArray: TFlyOSDArray; ArrayCount: integer; var FlyHeader: TFlyOSDHeader; FormatVersion: integer; ZEXML: TZEXMLSS; const FileName: string);
procedure SaveToXmlSS(const FileName: string; ZEXML: TZEXMLSS; FileFormat: integer);
function SaveToKML(const FileName: string;  ZEXML: TZEXMLSS; Memo: TMemo): boolean;

implementation

function OnlyDots(const st: string): string;
var
  i: integer;

begin
  result := '';
  for i := 1 to length(st) do
    if (st[i] = ',') then
      result := result + '.'
    else
      result := result + st[i];
end;

function IntToStrN(value: integer; NullCount: integer): string;
var
  t: integer;
  k: integer;

begin
  t := value;
  k := 0;
  if (t = 0) then
    k := 1;
  while t > 0 do
  begin
    inc(k);
    t := t div 10;
  end;
  result := IntToStr(value);
  for t := 1 to (NullCount - k) do
    result := '0' + result;
end;

//Читат *.r из потока
//INPUT
//      Stream: TStream           - поток
//  var FlyArray: TFlyOSDArray    - возвращаемый массив
//  var Count: integer            - кол-во записей
//  var FlyHeader: TFlyOSDHeader  - заголовок
//  var FormatVersion: integer    - версия формата
//  Memo: TMemo                   - мемо для сообщенй (если nil, то сообщения не выводятся)
//RETURN
//      byte - 0 - ok
function ReadFlyOSDR(Stream: TStream; var  FlyArray: TFlyOSDArray; var Count: integer; var FlyHeader: TFlyOSDHeader; var FormatVersion: integer; Memo: TMemo): byte; overload;
var
  HH: array[0..1] of byte;
  MaxCount: integer;
  sizeOSD: integer;
  readsize: integer;
  i: integer;
  tmpz: array[0..511] of byte;
  v: byte;

  procedure GrowArray();
  begin
    if (Count >= MaxCount) then
    begin
      inc(MaxCount, 512);
      SetLength(FlyArray, MaxCount);
    end;
  end;

  procedure WriteMemo(const txt: string);
  begin
    if (Assigned(Memo)) then
      Memo.Lines.Add(txt);
  end;

begin
  result := 0;
  Count := 0;
  MaxCount := 512;
  SetLength(FlyArray, MaxCount);
  FillChar(FlyHeader, sizeof(FlyHeader), 0);
  try
    if (Stream.Read(HH, 2) <> 2) then
    begin
      result := 1;
      WriteMemo('Error: Wrong length!');
      exit;
    end;

    sizeOSD := sizeof(word);
    sizeOSD := sizeof(FlyHeader);
    sizeOSD := sizeof(longint);
    sizeOSD := sizeof(FlyArray[0]);
    v := HH[0];
    FormatVersion := v;

    case (v) of
      1:
        begin
          if (Stream.Read(FlyArray[0], sizeOSD) <> sizeOSD) then
          begin
            result := 1;
            WriteMemo('Error: Wrong length!');
            exit;
          end;
          FlyHeader.time_hour := FlyArray[0].DateTime[0];
          FlyHeader.time_minute := FlyArray[0].DateTime[1];
          Stream.Position := 0;
        end;
      2:
        begin
          if (Stream.Read(FlyHeader, sizeof(FlyHeader)) <> sizeof(FlyHeader)) then
          begin
            result := 1;
            WriteMemo('Error: Wrong length!');
            exit;
          end;
          Stream.Read(tmpz, 493);
        end;
      else
        WriteMemo('Warning: Unknown version (try as v1)!');
    end;

    repeat
      readsize := Stream.Read(HH, 2);
      if (readsize = 0) then
        break;
      if (readsize <> 2) then
      begin
        result := 1;
        WriteMemo('Error: Wrong length!');
        exit;
      end;

      if ((HH[0] = v) and (HH[1] <= 17)) then
      begin
        for i := 1 to HH[1] do
        begin

          readsize := Stream.Read(FlyArray[Count], sizeOSD);
          if (readsize <> sizeOSD) then
          begin
             result := 1;
             WriteMemo('Error: Wrong length!');
             exit;
          end;
          inc(Count);
          GrowArray();
        end;
        readsize := Stream.Read(tmpz, 510 - sizeOSD*HH[1]);
      end
      else
      begin
        readsize := Stream.Read(tmpz, 510);
        if (readsize <> 510) then
        begin
          result := 1;
          WriteMemo('Error: Wrong length!');
          exit;
        end;
      end;
    until false;
  finally
    SetLength(FlyArray, Count);
  end;
end; //ReadFlyOSDR

//Читат *.r из файла
//INPUT
//  const FileName: string        - имя файла
//  var FlyArray: TFlyOSDArray    - возвращаемый массив
//  var Count: integer            - кол-во записей
//  var FlyHeader: TFlyOSDHeader  - заголовок
//  var FormatVersion: integer    - версия формата
//  Memo: TMemo                   - мемо для сообщенй (если nil, то сообщения не выводятся)
//RETURN
//      byte - 0 - ok
function ReadFlyOSDR(const FileName: string; var FlyArray: TFlyOSDArray; var Count: integer; var FlyHeader: TFlyOSDHeader; var FormatVersion: integer; Memo: TMemo): byte; overload;
var
  Stream :TStream;

begin
  result := 0;
  Stream := nil;
  try
    Stream := TFileStream.Create(FileName, fmOpenRead);
    result := ReadFlyOSDR(Stream, FlyArray, Count, FlyHeader, FormatVersion, Memo);
  finally
    if (Assigned(Stream)) then
      FreeAndNil(Stream);
  end;
end; //ReadFlyOSDR

//Сохраняет данные в формате excel xml spreadsheet
//INPUT
//  var FlyArray: TFlyOSDArray   - массив с данными
//      ArrayCount: integer      - кол-во элементов в массиве
//  var FlyHeader: TFlyOSDHeader - заголовок
//      FormatVersion: integer   - версия формата
//      ZEXML: TZEXMLSS          - хранилище
//  const FileName: string       - имя r-файла
procedure ArrayToXMLSS(var FlyArray: TFlyOSDArray; ArrayCount: integer; var FlyHeader: TFlyOSDHeader; FormatVersion: integer; ZEXML: TZEXMLSS; const FileName: string);
var
  i, j, n: integer;
  s: string;

begin
  ZEXML.Sheets.Count := 1;
  ZEXML.Sheets[0].Clear();
  ZEXML.Sheets.Count := 1;
  ZEXML.Styles.Count := 5;
  //заголовок
  ZEXML.Styles[0].Font.Size := 10;
  ZEXML.Styles[0].Font.Style := [fsBold];
  ZEXML.Styles[0].Alignment.Horizontal := ZHLeft;
  ZEXML.Styles[0].Alignment.Vertical := ZVCenter;
  //таблица
  ZEXML.Styles[1].Border[0].Weight := 1;
  ZEXML.Styles[1].Border[0].LineStyle := ZEContinuous;
  for i := 1 to 3 do
    ZEXML.Styles[1].Border[i].Assign(ZEXML.Styles[1].Border[0]);
  //заголовок таблицы
  ZEXML.Styles[2].Assign(ZEXML.Styles[1]);
  ZEXML.Styles[2].Font.Style := [fsBold];
  ZEXML.Styles[2].Alignment.Horizontal := ZHCenter;

  ZEXML.Styles[3].Assign(ZEXML.Styles[0]);
  ZEXML.Styles[3].Font.Style := [];
  ZEXML.Styles[3].Alignment.Horizontal := ZHRight;

  ZEXML.Styles[4].Assign(ZEXML.Styles[2]);
  ZEXML.Styles[4].Alignment.Horizontal := ZHLeft;

  with (ZEXML.Sheets[0]) do
  begin
    ColCount := 15;
    RowCount := ArrayCount  + 20;
    SheetOptions.PortraitOrientation := false;
    SheetOptions.MarginBottom := 10;
    SheetOptions.MarginLeft := 10;
    SheetOptions.MarginRight := 10;
    SheetOptions.MarginTop := 10;
    Cell[0, 0].Data := 'File Name:';
    Cell[1, 0].Data := FileName;
    MergeCells.AddRectXY(1, 0, 12, 0);
    Cell[0, 2].Data := 'Version:';
    Cell[1, 2].Data := IntToStr(FormatVersion);
    if (FormatVersion >= 2) then
    begin
      Cell[0, 3].Data := 'iSWVer:';
      Cell[1, 3].Data := IntToStr(FlyHeader.iSWVer);
      Cell[0, 4].Data := 'iHWVer:';
      Cell[1, 4].Data := IntToStr(FlyHeader.iHWVer);
      Cell[0, 5].Data := 'Date:';
      Cell[1, 5].Data := IntToStrN(FlyHeader.time_year, 4) + '.' + IntToStrN(FlyHeader.time_month, 2) + '.' + IntToStrN(FlyHeader.time_day, 2);
      Cell[0, 7].Data := 'Time Zone:';
      Cell[1, 7].Data := IntToStr(FlyHeader.timezone);
    end;
    Cell[0, 6].Data := 'Time:';
    Cell[1, 6].Data := IntToStrN(FlyHeader.time_hour, 2) + ':' + IntToStrN(FlyHeader.time_minute, 2) + ':' + IntToStrN(FlyHeader.time_second, 2);

    for i := 0 to 7  do
    begin
      Cell[0, i].CellStyle := 0;
      Cell[1, i].CellStyle := 3;
    end;
    Cell[1, 0].CellStyle := -1;

    Columns[0].WidthMM := 20;
    Columns[1].WidthMM := 20;
    Columns[4].WidthMM := 25;
    Columns[5].WidthMM := 15;
    Columns[6].WidthMM := 15;
    for i := 7 to 10 do
      Columns[i].WidthMM := 23;
    Columns[9].WidthMM := 20;
    Columns[11].WidthMM := 20;

    Cell[0, 9].Data := 'Time';
    Cell[1, 9].Data := 'State';
    Cell[2, 9].Data := 'Latitude';
    Cell[3, 9].Data := 'longitude';
    Cell[4, 9].Data := 'MSL_Altitude';
    Cell[5, 9].Data := 'Speed';
    Cell[6, 9].Data := 'Azimuth';
    Cell[7, 9].Data := 'Temperature';
    Cell[8, 9].Data := 'RotateSpeed';
    Cell[9, 9].Data := 'SYSBatVol';
    Cell[10, 9].Data := 'SYSBatVideo';
    Cell[11, 9].Data := 'SYSBatCur';
    Cell[12, 9].Data := 'FlyMode';

    for j := 0 to 12 do
      Cell[j, 9].CellStyle := 2;
    for i := 0 to ArrayCount - 1 do
    begin
      n := 10 + i;
      s := IntToStrN(FlyArray[i].DateTime[0], 2);
      for j := 1 to 2 do
        s := s + ':'  + IntToStrN(FlyArray[i].DateTime[j], 2);
      Cell[0, n].Data := s;
      Cell[1, n].Data := IntToStr(FlyArray[i].State);
      Cell[2, n].Data := OnlyDots(FloatToStr(FlyArray[i].Latitude / 1000000));
      Cell[3, n].Data := OnlyDots(FloatToStr(FlyArray[i].longitude / 1000000));
      Cell[4, n].Data := OnlyDots(FloatToStr(FlyArray[i].MSL_Altitude / 10));
      Cell[5, n].Data := OnlyDots(FloatToStr(FlyArray[i].Speed / 10));
      Cell[6, n].Data := OnlyDots(FloatToStr(FlyArray[i].Azimuth / 10));
      j := FlyArray[i].Temperature;
      if (FlyArray[i].Temperature > 200) then
        j := j - 255;
      Cell[7, n].Data := IntToStr(j);
      Cell[8, n].Data := IntToStr(FlyArray[i].RotateSpeed);
      Cell[9, n].Data := OnlyDots(FloatToStr(FlyArray[i].SYSBatVol / 10));
      Cell[10, n].Data := OnlyDots(FloatToStr(FlyArray[i].SYSBatVideo / 10));
      Cell[11, n].Data := OnlyDots(FloatToStr(FlyArray[i].SYSBatCur / 10));
      Cell[12, n].Data := IntToStr(FlyArray[i].FlyMode);

      for j := 1 to 12 do
        Cell[j, n].CellType := ZENumber;

      for j := 0 to 12 do
        Cell[j, n].CellStyle := 1;
    end; //for

    if (ArrayCount > 3) then
    begin
      n := 10 + ArrayCount;
      Cell[0, n].Data := 'MIN';
      Cell[0, n + 1].Data := 'MAX';
      Cell[0, n + 2].Data := 'AVG';

      for j := 0 to 2 do
      begin
        MergeCells.AddRectXY(0, n + j, 1, n + j);
        Cell[0, n + j].CellStyle := 4;
      end;

      for i := 2 to 12 do
      begin
        for j := 0 to 2 do
        begin
          Cell[i, n + j].CellStyle := 1;
          Cell[i, n + j].CellType := ZENumber;
        end;
        Cell[i, n].Formula := '=MIN(R[-' + IntToSTr(ArrayCount) + ']C:R[-1]C)';
        Cell[i, n + 1].Formula := '=MAX(R[-' + IntToSTr(ArrayCount+1) + ']C:R[-2]C)';
        Cell[i, n + 2].Formula := '=AVERAGE(R[-' + IntToSTr(ArrayCount+2) + ']C:R[-3]C)';
      end;
    end; //if
  end; //with
end; //ArrayToXMLSS

//Сохраняет в excel xml spreadsheet
procedure SaveToXmlSS(const FileName: string; ZEXML: TZEXMLSS; FileFormat: integer);
  procedure ChechForODS();
  var
    i, j, n: integer;

  begin
    with ZEXML.Sheets[0] do
    begin
      n := RowCount - 30 ;
      if (n < 0) then
        n := 1;
      for i := RowCount - 1 downto n do
      for j := 1 to 14 do
        if (Cell[j, i].Formula <> '') then
          Cell[j, i].Formula := ZER1C1ToA1(Cell[j, i].Formula, j, i, 0);
    end;
  end;

  procedure ChechForXML();
  var
    i, j, n: integer;

  begin
    with ZEXML.Sheets[0] do
    begin
      n := RowCount - 30 ;
      if (n < 0) then
        n := 1;
      for i := RowCount - 1 downto n do
      for j := 1 to 14 do
        if (Cell[j, i].Formula <> '') then
          Cell[j, i].Formula := ZEA1ToR1C1(Cell[j, i].Formula, j, i, 0);
    end;
  end;

begin
  case FileFormat of
    1: //ods
      begin
        ChechForODS();
        SaveXmlssToODFS(ZEXML, FileName, [0], [], nil, 'UTF-8');
      end;
    else //excel xml
      begin
        ChechForXML();
        SaveXmlssToEXML(ZEXML, FileName, [0], [], nil, 'UTF-8');
      end;
  end;
end;

function SaveToKML(const FileName: string; ZEXML: TZEXMLSS; Memo: TMemo): boolean;
var
  _xml: TZsspXMLWriterH;
  i: integer;
  s, slast: string;

  procedure _addDescription(const Name, Descr: string);
  begin
    _xml.WriteTag('name', Name);
    if (length(Descr) > 0) then
    begin
      _xml.WriteTagNode('description', true, true);
      _xml.WriteCDATA(Descr, false, true);
      _xml.WriteEndTagNode(); //description
    end;
  end;

  //Добавить Placemark point
  procedure _AddPointPlaceMark(const Name, Descr, Coords: string);
  begin
    _xml.WriteTagNode('Placemark', true, true);
    _addDescription(Name, Descr);
    _xml.WriteTagNode('Point', true, true);
    _xml.WriteTag('coordinates', Coords);
    _xml.WriteEndTagNode(); //Point
    _xml.WriteEndTagNode(); //Placemark
  end; //_AddPointPlaceMark

begin
  _xml := nil;
  result := false;
  try
    try
      _xml := TZsspXMLWriterH.Create();
      _xml.TabLength := 1;
      _xml.TabSymbol := ' ';
      if (not _xml.BeginSaveToFile(FileName)) then
      begin
        if (Assigned(Memo)) then
          Memo.Lines.Add('Can not save to ' + FileName);
        exit;
      end;

      _xml.Attributes.Add('version', '1.0');
      _xml.Attributes.Add('encoding', 'UTF-8');
      _xml.WriteInstruction('xml', false);
      _xml.Attributes.Clear();
      _xml.Attributes.Add('xmlns', 'http://www.opengis.net/kml/2.2');
      _xml.WriteTagNode('kml', true, true);
      _xml.Attributes.Clear();
      _xml.WriteTagNode('Document', true, true);
      _addDescription('Flight path', 'Start point, flight path and stop point.');

      //first
      s := ZEXML.Sheets[0].Cell[0, 10].Data;
      s := 'Start time: ' + s + '<br>';
      if (length(ZEXML.Sheets[0].Cell[1, 5].Data) > 0) then
        s := s + 'Date: ' + ZEXML.Sheets[0].Cell[1, 5].Data + '<br>';
      with ZEXML.Sheets[0] do
        _AddPointPlaceMark('Start point', s, Cell[3, 10].Data + ',' + Cell[2, 10].Data + ',' + Cell[4, 10].Data);

      _xml.WriteTagNode('Placemark', true, true);
      _addDescription('Flight path', '');
      _xml.WriteTagNode('LineString', true, true);
      _xml.WriteTag('extrude', '0');
      _xml.WriteTag('tesselate', '0');
      _xml.WriteTag('altitudeMode', 'absolute');

      s := '';
      slast := '';
      for i := 10 to ZEXML.Sheets[0].RowCount - 1 do
      begin
        if (ZEXML.Sheets[0].Cell[0, i].Data = 'MIN') then
          break;
        with ZEXML.Sheets[0] do
          slast := Cell[3, i].Data + ',' + Cell[2, i].Data + ',' + Cell[4, i].Data;
          s := s + slast + ' ';
      end;

      _xml.WriteTag('coordinates', s);

      _xml.WriteEndTagNode(); //LineString
      _xml.WriteEndTagNode(); //Placemark

      s := 'Stop time: ' + ZEXML.Sheets[0].Cell[0, i - 1].Data;
      _AddPointPlaceMark('Stop point', s, slast);

      _xml.WriteEndTagNode(); //Document
      _xml.WriteEndTagNode(); //kml
      _xml.EndSaveTo();
      result := true;
    except
      on E: exception do
       if (Assigned(Memo)) then
         Memo.Lines.Add('Error (kml): ' + e.Message);
    end;
  finally
    if (Assigned(_xml)) then
      FreeAndNil(_xml);
  end;
end;


end.

